#!/usr/bin/perl
##############################################################################
# WWWBoard                       Version 2.0 ALPHA 2.1                       #
# Copyright 1996                 worldwidemart.com                           #
# Created 10/21/95               Last Modified 01/07/00                      #
# Scripts Archive:               https://www.worldwidemart.com/scripts/          #
##############################################################################
# COPYRIGHT NOTICE                                                           #
# This script is distributed under the Artistic License.                     #
# You may freely use and modify it, keeping this copyright notice intact.    #
##############################################################################

use strict;
use warnings;
use CGI qw(:standard);
use CGI::Carp qw(fatalsToBrowser);
use Fcntl qw(:flock);

# Configuration
my $basedir     = '.';                      # Base directory
my $baseurl     = '/wwwboard';              # Base URL
my $cgi_url     = '/cgi-bin/wwwboard.pl';   # Script URL
my $meession    = "$basedir/wwwboard.html"; # Main board file
my $message_dir = "$basedir/messages";       # Directory for message files
my $data_file   = "$basedir/data.txt";       # Data file
my $max_posts   = 500;                       # Maximum posts to keep

# Admin settings
my $admin_password = 'changeme';             # CHANGE THIS!

##############################################################################
# Main Script
##############################################################################

my $cgi = CGI->new;
my $action = $cgi->param('action') || 'view';

if ($action eq 'post') {
    post_message();
} elsif ($action eq 'followup') {
    show_followup_form();
} elsif ($action eq 'delete' && verify_admin()) {
    delete_message();
} else {
    show_board();
}

##############################################################################
# Subroutines
##############################################################################

sub show_board {
    print $cgi->header('text/html');

    if (-e $meession) {
        open(my $fh, '<', $meession) or die "Cannot open board: $!";
        print while <$fh>;
        close($fh);
    } else {
        print_default_board();
    }
}

sub post_message {
    my $name     = $cgi->param('name') || 'Anonymous';
    my $email    = $cgi->param('email') || '';
    my $subject  = $cgi->param('subject') || 'No Subject';
    my $body     = $cgi->param('body') || '';
    my $followup = $cgi->param('followup') || 0;

    # Validate
    unless ($body) {
        error_page("Message body is required.");
        return;
    }

    # Sanitize
    $name    = $cgi->escapeHTML($name);
    $email   = $cgi->escapeHTML($email);
    $subject = $cgi->escapeHTML($subject);
    $body    = $cgi->escapeHTML($body);
    $body    =~ s/\n/<br>\n/g;

    # Get next message number
    my $num = get_next_num();

    # Create message file
    create_message_file($num, $name, $email, $subject, $body, $followup);

    # Update main board
    update_board($num, $name, $subject, $followup);

    # Redirect to board
    print $cgi->redirect($meession);
}

sub get_next_num {
    my $num = 1;

    if (-e $data_file) {
        open(my $fh, '<', $data_file);
        $num = <$fh>;
        close($fh);
        chomp($num);
        $num++;
    }

    open(my $fh, '>', $data_file);
    print $fh $num;
    close($fh);

    return $num;
}

sub create_message_file {
    my ($num, $name, $email, $subject, $body, $followup) = @_;

    mkdir $message_dir unless -d $message_dir;

    my $timestamp = localtime();
    my $email_link = $email ? qq{<a href="mailto:$email">$name</a>} : $name;

    open(my $fh, '>', "$message_dir/$num.html") or die "Cannot create message: $!";
    print $fh <<HTML;
<!DOCTYPE html>
<html>
<head>
    <title>$subject</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; }
        .message { background: #f8f9fa; padding: 20px; border-radius: 5px; margin: 20px 0; }
        .meta { color: #666; font-size: 0.9em; }
        .actions { margin-top: 20px; }
        .actions a { margin-right: 15px; }
    </style>
</head>
<body>
    <h1>$subject</h1>
    <div class="meta">
        Posted by: $email_link<br>
        Date: $timestamp
    </div>
    <div class="message">
        $body
    </div>
    <div class="actions">
        <a href="$cgi_url?action=followup&followup=$num">Post Followup</a>
        <a href="$meession">Back to Board</a>
    </div>
</body>
</html>
HTML
    close($fh);
}

sub update_board {
    my ($num, $name, $subject, $followup) = @_;

    my $entry = qq{<li><a href="$message_dir/$num.html">$subject</a> - $name</li>\n};

    my $content = '';
    if (-e $meession) {
        open(my $fh, '<', $meession);
        local $/;
        $content = <$fh>;
        close($fh);
    } else {
        $content = get_board_template();
    }

    my $marker = '<!-- MESSAGES -->';
    $content =~ s/(\Q$marker\E)/$1\n$entry/;

    open(my $fh, '>', $meession);
    print $fh $content;
    close($fh);
}

sub show_followup_form {
    my $followup = $cgi->param('followup') || 0;

    print $cgi->header('text/html');
    print <<HTML;
<!DOCTYPE html>
<html>
<head>
    <title>Post Followup</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto; padding: 20px; }
        .form-group { margin-bottom: 15px; }
        label { display: block; margin-bottom: 5px; font-weight: bold; }
        input, textarea { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 4px; box-sizing: border-box; }
        textarea { height: 200px; }
        button { background: #007bff; color: white; padding: 12px 24px; border: none; border-radius: 4px; cursor: pointer; }
    </style>
</head>
<body>
    <h1>Post Followup</h1>
    <form action="$cgi_url" method="POST">
        <input type="hidden" name="action" value="post">
        <input type="hidden" name="followup" value="$followup">
        <div class="form-group">
            <label>Name</label>
            <input type="text" name="name" required>
        </div>
        <div class="form-group">
            <label>Email</label>
            <input type="email" name="email">
        </div>
        <div class="form-group">
            <label>Subject</label>
            <input type="text" name="subject" value="Re: " required>
        </div>
        <div class="form-group">
            <label>Message</label>
            <textarea name="body" required></textarea>
        </div>
        <button type="submit">Post Message</button>
    </form>
</body>
</html>
HTML
}

sub verify_admin {
    my $pass = $cgi->param('password') || '';
    return $pass eq $admin_password;
}

sub delete_message {
    my $num = $cgi->param('num') || 0;
    unlink "$message_dir/$num.html" if $num && -e "$message_dir/$num.html";
    print $cgi->redirect($meession);
}

sub error_page {
    my ($msg) = @_;
    print $cgi->header('text/html');
    print "<html><body><h1>Error</h1><p>$msg</p><p><a href='javascript:history.back()'>Go Back</a></p></body></html>";
}

sub print_default_board {
    print get_board_template();
}

sub get_board_template {
    return <<'TEMPLATE';
<!DOCTYPE html>
<html>
<head>
    <title>WWWBoard - Discussion Forum</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 800px; margin: 0 auto; padding: 20px; }
        h1 { color: #333; }
        .post-link { background: #28a745; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; }
    </style>
</head>
<body>
    <h1>WWWBoard - Discussion Forum</h1>
    <p><a href="wwwboard.pl?action=followup" class="post-link">Post New Message</a></p>
    <hr>
    <h2>Messages</h2>
    <ul>
    <!-- MESSAGES -->
    </ul>
</body>
</html>
TEMPLATE
}

__END__

=head1 NAME

wwwboard.pl - Threaded Discussion Forum

=head1 DESCRIPTION

A threaded World Wide Web discussion forum and message board that allows
users to post new messages and follow up to existing ones.

=head1 LICENSE

Artistic License

=cut
