#!/usr/bin/perl
##############################################################################
# Text Counter                      Version 1.3                              #
# Copyright 1996                    worldwidemart.com                        #
# Scripts Archive:                  https://www.worldwidemart.com/scripts/       #
##############################################################################
# COPYRIGHT NOTICE                                                           #
# This script is distributed under the Artistic License.                     #
# You may freely use and modify it, keeping this copyright notice intact.    #
##############################################################################
# DESCRIPTION                                                                #
# A simple text-based page counter that outputs plain text instead of        #
# graphical images. Lighter weight than graphical counters.                  #
##############################################################################

use strict;
use warnings;
use CGI qw(:standard);
use Fcntl qw(:flock);

# Configuration
my $count_file    = './count.txt';     # File to store count
my $start_count   = 0;                 # Starting count
my $pad_length    = 6;                 # Minimum digits (zero-padded)
my $show_commas   = 0;                 # Add commas to large numbers
my $prefix        = '';                # Text before count
my $suffix        = '';                # Text after count
my $count_unique  = 0;                 # Only count unique IPs

# Lock settings
my $lock_file     = './count.lock';
my $max_wait      = 5;                 # Maximum seconds to wait for lock

# IP tracking for unique counts
my $ip_file       = './visitors.txt';
my $ip_expire     = 3600;              # Seconds before IP can be counted again

##############################################################################
# Main Script
##############################################################################

my $cgi = CGI->new;

# Print header
print $cgi->header('text/plain');

# Get current count
my $count = get_count();

# Check if we should increment
my $should_increment = 1;

if ($count_unique) {
    $should_increment = check_unique_visitor();
}

# Increment and save
if ($should_increment) {
    $count++;
    save_count($count);

    if ($count_unique) {
        log_visitor();
    }
}

# Format output
my $output = format_count($count);
print $prefix . $output . $suffix;

##############################################################################
# Subroutines
##############################################################################

sub get_count {
    if (-e $count_file) {
        open(my $fh, '<', $count_file) or return $start_count;
        my $count = <$fh>;
        close($fh);
        chomp($count) if $count;
        return $count || $start_count;
    }
    return $start_count;
}

sub save_count {
    my ($count) = @_;

    # Acquire lock
    open(my $lock_fh, '>', $lock_file) or return;
    my $waited = 0;
    while (!flock($lock_fh, LOCK_EX | LOCK_NB)) {
        sleep(1);
        $waited++;
        if ($waited >= $max_wait) {
            close($lock_fh);
            return;
        }
    }

    # Write count
    open(my $fh, '>', $count_file) or do {
        close($lock_fh);
        return;
    };
    print $fh $count;
    close($fh);

    # Release lock
    flock($lock_fh, LOCK_UN);
    close($lock_fh);
}

sub format_count {
    my ($count) = @_;

    # Zero-pad if needed
    if ($pad_length > 0) {
        $count = sprintf("%0${pad_length}d", $count);
    }

    # Add commas
    if ($show_commas && $count >= 1000) {
        $count = reverse($count);
        $count =~ s/(\d{3})(?=\d)/$1,/g;
        $count = reverse($count);
    }

    return $count;
}

sub check_unique_visitor {
    my $ip = $ENV{REMOTE_ADDR} || '127.0.0.1';
    my $now = time();

    return 1 unless -e $ip_file;

    open(my $fh, '<', $ip_file) or return 1;
    while (<$fh>) {
        chomp;
        my ($stored_ip, $timestamp) = split(/\|/);
        if ($stored_ip eq $ip && ($now - $timestamp) < $ip_expire) {
            close($fh);
            return 0;  # Don't count this visitor
        }
    }
    close($fh);

    return 1;  # Count this visitor
}

sub log_visitor {
    my $ip = $ENV{REMOTE_ADDR} || '127.0.0.1';
    my $now = time();

    # Read existing entries, filtering expired
    my @entries;
    if (-e $ip_file) {
        open(my $fh, '<', $ip_file) or return;
        while (<$fh>) {
            chomp;
            my ($stored_ip, $timestamp) = split(/\|/);
            if (($now - $timestamp) < $ip_expire) {
                push @entries, $_;
            }
        }
        close($fh);
    }

    # Add current visitor
    push @entries, "$ip|$now";

    # Write back
    open(my $fh, '>', $ip_file) or return;
    print $fh join("\n", @entries) . "\n";
    close($fh);
}

__END__

=head1 NAME

textcounter.pl - Text-Based Page Counter

=head1 SYNOPSIS

  <!--#include virtual="/cgi-bin/textcounter.pl" -->

  Or in HTML:
  <script>
    document.write('<iframe src="/cgi-bin/textcounter.pl" frameborder="0"></iframe>');
  </script>

=head1 DESCRIPTION

A lightweight text-based page counter. Unlike graphical counters, this
outputs plain text numbers, making it faster and easier to style with CSS.

=head1 CONFIGURATION

$count_file   - Path to file storing the count
$start_count  - Initial count value
$pad_length   - Zero-pad to this many digits (0 to disable)
$show_commas  - Add thousands separators
$count_unique - Only count unique IP addresses

=head1 MODERN ALTERNATIVES

For modern page counting:

Server-side:
* Google Analytics (https://analytics.google.com)
* Plausible (https://plausible.io)
* Fathom (https://usefathom.com)
* Matomo (https://matomo.org)

If you need visible counters:
* Use analytics API to fetch real counts
* Store counts in a database
* Use serverless functions (Vercel, Netlify)

Example with localStorage (client-side demo):
  let visits = localStorage.getItem('visits') || 0;
  visits++;
  localStorage.setItem('visits', visits);
  document.getElementById('counter').textContent = visits;

=head1 LICENSE

Artistic License

=cut
