#!/usr/bin/perl
##############################################################################
# SSI Random Image                  Version 1.1                              #
# Copyright 1996                    worldwidemart.com                        #
# Scripts Archive:                  https://www.worldwidemart.com/scripts/       #
##############################################################################
# COPYRIGHT NOTICE                                                           #
# This script is distributed under the Artistic License.                     #
# You may freely use and modify it, keeping this copyright notice intact.    #
##############################################################################
# DESCRIPTION                                                                #
# A Server Side Include (SSI) optimized version of Random Image Displayer.   #
# Returns just the filename or HTML for embedding via SSI.                   #
##############################################################################

use strict;
use warnings;

# Configuration
my $image_dir  = './images';           # Directory containing images
my $image_url  = '/images';            # URL path to images
my @extensions = qw(.gif .jpg .jpeg .png);  # Valid image extensions

# Output options
my $output_html = 1;                   # 1 = output <img> tag, 0 = just filename
my $img_border  = 0;                   # Image border
my $img_alt     = '';                  # Alt text (empty = use filename)

# Optional: Link images
my $link_image  = 0;                   # Wrap image in link
my $link_url    = '';                  # Link URL (empty = link to image itself)

##############################################################################
# Main Script
##############################################################################

# Get image list
my @images = get_images();

unless (@images) {
    print "Content-type: text/html\n\n";
    print "<!-- No images found -->";
    exit;
}

# Select random image
srand();
my $random_image = $images[int(rand(@images))];
my $full_url = "$image_url/$random_image";

# Output
print "Content-type: text/html\n\n";

if ($output_html) {
    my $alt = $img_alt || $random_image;
    my $img_tag = qq{<img src="$full_url" alt="$alt" border="$img_border">};

    if ($link_image) {
        my $href = $link_url || $full_url;
        print qq{<a href="$href">$img_tag</a>};
    } else {
        print $img_tag;
    }
} else {
    print $random_image;
}

##############################################################################
# Subroutines
##############################################################################

sub get_images {
    my @found;

    opendir(my $dh, $image_dir) or return @found;
    while (my $file = readdir($dh)) {
        next if $file =~ /^\./;
        next unless -f "$image_dir/$file";

        foreach my $ext (@extensions) {
            if ($file =~ /\Q$ext\E$/i) {
                push @found, $file;
                last;
            }
        }
    }
    closedir($dh);

    return @found;
}

__END__

=head1 NAME

ssi_rand_image.pl - SSI-Optimized Random Image Script

=head1 SYNOPSIS

  <!--#include virtual="/cgi-bin/ssi_rand_image.pl" -->

=head1 DESCRIPTION

This script is optimized for Server Side Includes (SSI). It outputs
HTML directly that can be embedded in a page via SSI directives.

Unlike the standard rand_image.pl which redirects, this outputs the
actual HTML img tag or filename for direct embedding.

=head1 CONFIGURATION

$image_dir   - Server path to images
$image_url   - URL path to images
$output_html - Output <img> tag (1) or just filename (0)
$link_image  - Wrap image in a link
$link_url    - URL for link (empty = link to image itself)

=head1 SSI REQUIREMENTS

Your server must have SSI enabled. For Apache, ensure:

  Options +Includes
  AddHandler server-parsed .shtml

Or to enable for all HTML files:

  AddOutputFilter INCLUDES .html

=head1 MODERN ALTERNATIVES

For modern implementations:

Server-side includes are rarely used today. Instead use:

* PHP includes
* JavaScript DOM manipulation
* Static site generators
* Template engines (Jinja, EJS, Handlebars)

JavaScript equivalent:
  <script>
    const images = ['img1.jpg', 'img2.jpg', 'img3.jpg'];
    document.write('<img src="/images/' +
      images[Math.floor(Math.random() * images.length)] +
    '" alt="Random">');
  </script>

=head1 LICENSE

Artistic License

=cut
