#!/usr/bin/perl
##############################################################################
# Random Text Generator            Version 1.2                               #
# Copyright 1996                   worldwidemart.com                         #
# Scripts Archive:                 https://www.worldwidemart.com/scripts/        #
##############################################################################
# COPYRIGHT NOTICE                                                           #
# This script is distributed under the Artistic License.                     #
# You may freely use and modify it, keeping this copyright notice intact.    #
##############################################################################
# DESCRIPTION                                                                #
# Displays a random text selection from a file or predefined list.           #
# Useful for random quotes, tips, fortunes, or taglines.                     #
##############################################################################

use strict;
use warnings;
use CGI qw(:standard);

# Configuration
my $text_file = './quotes.txt';        # File containing text entries
my $separator = '%%';                   # Entry separator in file

# Text can also be defined directly here
my @texts = ();  # Leave empty to use file
# my @texts = (
#     "Quote one goes here.",
#     "Quote two goes here.",
#     "Quote three goes here.",
# );

# Output options
my $output_html = 1;                   # 1 = HTML, 0 = plain text
my $wrap_tag    = 'blockquote';        # HTML tag to wrap text (empty for none)
my $css_class   = 'random-text';       # CSS class for wrapper

##############################################################################
# Main Script
##############################################################################

my $cgi = CGI->new;

# Load texts from file if array is empty
if (!@texts && -e $text_file) {
    @texts = load_texts($text_file);
}

unless (@texts) {
    print $cgi->header('text/plain');
    print "No texts configured.";
    exit;
}

# Select random text
srand();
my $random_text = $texts[int(rand(@texts))];

# Output
if ($output_html) {
    print $cgi->header('text/html');
    if ($wrap_tag) {
        print qq{<$wrap_tag class="$css_class">$random_text</$wrap_tag>};
    } else {
        print $random_text;
    }
} else {
    print $cgi->header('text/plain');
    print $random_text;
}

##############################################################################
# Subroutines
##############################################################################

sub load_texts {
    my ($file) = @_;
    my @loaded;

    open(my $fh, '<', $file) or return @loaded;
    my $content = do { local $/; <$fh> };
    close($fh);

    # Split by separator
    @loaded = split(/\n\Q$separator\E\n/, $content);

    # Clean up entries
    @loaded = grep { /\S/ } @loaded;  # Remove empty entries
    s/^\s+|\s+$//g for @loaded;       # Trim whitespace

    return @loaded;
}

__END__

=head1 NAME

rand_text.pl - Random Text/Quote Generator

=head1 SYNOPSIS

  <!--#include virtual="/cgi-bin/rand_text.pl" -->

=head1 DESCRIPTION

Displays a random text selection from a file or predefined list.
Commonly used for random quotes, tips of the day, fortunes, or taglines.

=head1 TEXT FILE FORMAT

Texts are separated by a line containing only "%%":

  First quote or text goes here.
  It can span multiple lines.
  %%
  Second quote goes here.
  %%
  Third quote goes here.

=head1 CONFIGURATION

$text_file   - Path to file containing texts
$separator   - Line that separates entries (default: %%)
@texts       - Array of texts (if not using file)
$output_html - Output as HTML (1) or plain text (0)
$wrap_tag    - HTML tag to wrap output
$css_class   - CSS class for wrapper element

=head1 HISTORICAL CONTEXT

Random quote/fortune scripts were extremely popular on 1990s websites.
The Unix "fortune" program inspired many web versions, displaying
random quotes, jokes, or wisdom on each page load.

=head1 MODERN ALTERNATIVES

For random text in modern web development:

JavaScript:
  const quotes = [
    "Quote one...",
    "Quote two...",
    "Quote three..."
  ];
  const random = quotes[Math.floor(Math.random() * quotes.length)];
  document.getElementById('quote').textContent = random;

Fetch from API:
  // Using a quotes API
  fetch('https://api.quotable.io/random')
    .then(r => r.json())
    .then(data => {
      document.getElementById('quote').textContent = data.content;
      document.getElementById('author').textContent = data.author;
    });

Quote APIs:
* Quotable (https://github.com/lukePeavey/quotable)
* They Said So (https://theysaidso.com/api)
* Forismatic (https://forismatic.com/en/api/)

=head1 LICENSE

Artistic License

=cut
