#!/usr/bin/perl
##############################################################################
# Random Link Generator            Version 2.0                               #
# Copyright 1996                   worldwidemart.com                         #
# Scripts Archive:                 https://www.worldwidemart.com/scripts/        #
##############################################################################
# COPYRIGHT NOTICE                                                           #
# This script is distributed under the Artistic License.                     #
# You may freely use and modify it, keeping this copyright notice intact.    #
##############################################################################
# DESCRIPTION                                                                #
# Selects and displays a random link from a predefined list. Can redirect    #
# directly or output HTML for embedding.                                     #
##############################################################################

use strict;
use warnings;
use CGI qw(:standard);

# Configuration
my $links_file = './links.txt';        # File containing links (optional)
my $output_mode = 'redirect';          # 'redirect', 'html', or 'url'

# Links can be defined here or loaded from file
# Format: URL|Title|Description
my @links = (
    'https://example.com|Example Site|An example website',
    'https://example.org|Example Org|Another example',
    'https://example.net|Example Net|Yet another example',
);

# HTML output settings
my $link_class  = '';                  # CSS class for link
my $target      = '';                  # Link target (_blank, etc.)
my $show_desc   = 0;                   # Show description after link

##############################################################################
# Main Script
##############################################################################

my $cgi = CGI->new;

# Check for mode parameter
$output_mode = $cgi->param('mode') if $cgi->param('mode');

# Load links from file if it exists
if (-e $links_file) {
    @links = load_links_file($links_file);
}

unless (@links) {
    print $cgi->header('text/plain');
    print "Error: No links configured.";
    exit;
}

# Select random link
my $random_link = $links[int(rand(@links))];
my ($url, $title, $desc) = split(/\|/, $random_link);
$title ||= $url;
$desc  ||= '';

# Output based on mode
if ($output_mode eq 'redirect') {
    print $cgi->redirect($url);
} elsif ($output_mode eq 'html') {
    print $cgi->header('text/html');
    my $attrs = qq{href="$url"};
    $attrs .= qq{ class="$link_class"} if $link_class;
    $attrs .= qq{ target="$target"} if $target;
    my $output = qq{<a $attrs>$title</a>};
    $output .= " - $desc" if $show_desc && $desc;
    print $output;
} elsif ($output_mode eq 'url') {
    print $cgi->header('text/plain');
    print $url;
} elsif ($output_mode eq 'json') {
    print $cgi->header('application/json');
    $title =~ s/"/\\"/g;
    $desc =~ s/"/\\"/g;
    print qq{{"url":"$url","title":"$title","description":"$desc"}};
}

##############################################################################
# Subroutines
##############################################################################

sub load_links_file {
    my ($file) = @_;
    my @loaded;

    open(my $fh, '<', $file) or return @loaded;
    while (<$fh>) {
        chomp;
        next if /^#/;      # Skip comments
        next if /^\s*$/;   # Skip empty lines
        push @loaded, $_;
    }
    close($fh);

    return @loaded;
}

__END__

=head1 NAME

rand_link.pl - Random Link Generator

=head1 SYNOPSIS

  <!-- Redirect to random link -->
  <a href="/cgi-bin/rand_link.pl">Random Site</a>

  <!-- Get HTML link -->
  <!--#include virtual="/cgi-bin/rand_link.pl?mode=html" -->

  <!-- Get just URL -->
  <!--#include virtual="/cgi-bin/rand_link.pl?mode=url" -->

=head1 DESCRIPTION

Selects and outputs a random link from a predefined list. Useful for
"random site" buttons, webrings, partner rotations, or link exchanges.

=head1 LINK FILE FORMAT

Links can be stored in an external file (links.txt) with format:
  URL|Title|Description

Example:
  https://example.com|Example Site|An example website
  https://example.org|Example Org|Another example

Lines starting with # are treated as comments.

=head1 OUTPUT MODES

=over 4

=item redirect - HTTP redirect to random URL (default)

=item html - Output <a> tag HTML

=item url - Output plain URL text

=item json - Output JSON object

=back

=head1 HISTORICAL CONTEXT

Random link scripts were popular in the 1990s for:
- Webrings (circular link networks)
- Link exchanges
- "Surprise me" buttons
- Banner ad rotation

=head1 MODERN ALTERNATIVES

For modern random links:

JavaScript:
  const links = [
    { url: 'https://example.com', title: 'Example' },
    { url: 'https://example.org', title: 'Another' }
  ];
  const random = links[Math.floor(Math.random() * links.length)];
  window.location.href = random.url;

Or as a link:
  <a href="#" onclick="window.location.href=links[Math.floor(Math.random()*links.length)].url; return false;">
    Random Site
  </a>

=head1 LICENSE

Artistic License

=cut
