#!/usr/bin/perl
##############################################################################
# Countdown                         Version 1.1                              #
# Copyright 1996                    worldwidemart.com                        #
# Scripts Archive:                  https://www.worldwidemart.com/scripts/       #
##############################################################################
# COPYRIGHT NOTICE                                                           #
# This script is distributed under the Artistic License.                     #
# You may freely use and modify it, keeping this copyright notice intact.    #
##############################################################################
# DESCRIPTION                                                                #
# Displays a countdown to a specified date/time. Can output as text or       #
# as an image using digit graphics.                                          #
##############################################################################

use strict;
use warnings;
use CGI qw(:standard);
use Time::Local;
use POSIX qw(strftime);

# Configuration
my $target_year   = 2000;     # Target year (4 digits)
my $target_month  = 1;        # Target month (1-12)
my $target_day    = 1;        # Target day (1-31)
my $target_hour   = 0;        # Target hour (0-23)
my $target_min    = 0;        # Target minute (0-59)
my $target_sec    = 0;        # Target second (0-59)

my $output_type   = 'text';   # 'text' or 'image'
my $digit_dir     = './digits'; # Directory containing digit images
my $digit_ext     = '.gif';    # Extension of digit images
my $separator     = ':';       # Separator between units

# Display options
my $show_years    = 1;
my $show_months   = 1;
my $show_days     = 1;
my $show_hours    = 1;
my $show_minutes  = 1;
my $show_seconds  = 1;

# Labels for text output
my %labels = (
    years   => 'years',
    months  => 'months',
    days    => 'days',
    hours   => 'hours',
    minutes => 'minutes',
    seconds => 'seconds',
);

##############################################################################
# Main Script
##############################################################################

my $cgi = CGI->new;

# Allow override via query parameters
$target_year  = $cgi->param('year')  if $cgi->param('year');
$target_month = $cgi->param('month') if $cgi->param('month');
$target_day   = $cgi->param('day')   if $cgi->param('day');
$target_hour  = $cgi->param('hour')  if $cgi->param('hour');
$target_min   = $cgi->param('min')   if $cgi->param('min');
$output_type  = $cgi->param('type')  if $cgi->param('type');

# Calculate time difference
my $target_time = eval {
    timelocal($target_sec, $target_min, $target_hour,
              $target_day, $target_month - 1, $target_year);
};

if ($@) {
    print $cgi->header('text/html');
    print "Error: Invalid target date specified.";
    exit;
}

my $now = time();
my $diff = $target_time - $now;

# Calculate units
my ($years, $months, $days, $hours, $minutes, $seconds) = calc_countdown($diff);

# Output based on type
if ($output_type eq 'image') {
    output_image($years, $months, $days, $hours, $minutes, $seconds);
} else {
    output_text($years, $months, $days, $hours, $minutes, $seconds);
}

##############################################################################
# Subroutines
##############################################################################

sub calc_countdown {
    my ($seconds_left) = @_;

    if ($seconds_left <= 0) {
        return (0, 0, 0, 0, 0, 0);
    }

    my $years = int($seconds_left / (365.25 * 24 * 60 * 60));
    $seconds_left -= $years * (365.25 * 24 * 60 * 60);

    my $months = int($seconds_left / (30.44 * 24 * 60 * 60));
    $seconds_left -= $months * (30.44 * 24 * 60 * 60);

    my $days = int($seconds_left / (24 * 60 * 60));
    $seconds_left -= $days * (24 * 60 * 60);

    my $hours = int($seconds_left / (60 * 60));
    $seconds_left -= $hours * (60 * 60);

    my $minutes = int($seconds_left / 60);
    my $seconds = int($seconds_left % 60);

    return ($years, $months, $days, $hours, $minutes, $seconds);
}

sub output_text {
    my ($years, $months, $days, $hours, $minutes, $seconds) = @_;

    print $cgi->header('text/html');

    my @parts;

    if ($years <= 0 && $months <= 0 && $days <= 0 &&
        $hours <= 0 && $minutes <= 0 && $seconds <= 0) {
        print "The countdown has ended!";
        return;
    }

    push @parts, "$years $labels{years}" if $show_years && $years > 0;
    push @parts, "$months $labels{months}" if $show_months && $months > 0;
    push @parts, "$days $labels{days}" if $show_days && $days > 0;
    push @parts, "$hours $labels{hours}" if $show_hours && $hours > 0;
    push @parts, "$minutes $labels{minutes}" if $show_minutes && $minutes > 0;
    push @parts, "$seconds $labels{seconds}" if $show_seconds;

    print join(", ", @parts);
}

sub output_image {
    my ($years, $months, $days, $hours, $minutes, $seconds) = @_;

    # Build the countdown string
    my $countdown_str = sprintf("%02d%s%02d%s%02d%s%02d",
        $days, $separator, $hours, $separator, $minutes, $separator, $seconds);

    print $cgi->header('text/html');

    print '<span class="countdown">';
    for my $char (split //, $countdown_str) {
        if ($char eq $separator) {
            print '<span class="separator">:</span>';
        } else {
            my $img = "$digit_dir/$char$digit_ext";
            print qq{<img src="$img" alt="$char">};
        }
    }
    print '</span>';
}

__END__

=head1 NAME

countdown.pl - Countdown Timer Script

=head1 SYNOPSIS

  <!-- Text output -->
  <script>
    document.write('<iframe src="/cgi-bin/countdown.pl?year=2025&month=1&day=1"></iframe>');
  </script>

  <!-- Or via SSI -->
  <!--#include virtual="/cgi-bin/countdown.pl?year=2025&month=1&day=1" -->

=head1 DESCRIPTION

Displays a countdown to a specified date/time. Originally popular for
Y2K countdowns in the late 1990s.

=head1 PARAMETERS

=over 4

=item year - Target year (4 digits)

=item month - Target month (1-12)

=item day - Target day (1-31)

=item hour - Target hour (0-23)

=item min - Target minute (0-59)

=item type - Output type ('text' or 'image')

=back

=head1 MODERN ALTERNATIVES

For modern countdowns, use JavaScript:

  function countdown(target) {
    const diff = new Date(target) - new Date();
    const days = Math.floor(diff / (1000 * 60 * 60 * 24));
    const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
    const mins = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
    const secs = Math.floor((diff % (1000 * 60)) / 1000);
    return `${days}d ${hours}h ${mins}m ${secs}s`;
  }

Or use libraries like:
- Moment.js / Day.js
- Countdown.js
- FlipClock.js

=head1 LICENSE

Artistic License

=cut
