#!/usr/bin/perl
##############################################################################
# Credit Card Verifier             Version 1.3                               #
# Copyright 1997                   worldwidemart.com                         #
# Scripts Archive:                 https://www.worldwidemart.com/scripts/        #
##############################################################################
# COPYRIGHT NOTICE                                                           #
# This script is distributed under the Artistic License.                     #
# You may freely use and modify it, keeping this copyright notice intact.    #
##############################################################################
# DESCRIPTION                                                                #
# Validates credit card numbers using the Luhn algorithm (mod 10 check).     #
# Also identifies card type based on number prefix and length.               #
##############################################################################
# SECURITY WARNING                                                           #
# This script only validates number FORMAT. It does NOT verify:              #
# - Whether the card actually exists                                         #
# - Whether the card has available credit                                    #
# - Whether the card is stolen or fraudulent                                 #
# For actual payment processing, use a proper payment gateway.               #
##############################################################################

use strict;
use warnings;
use CGI qw(:standard);

##############################################################################
# Main Script
##############################################################################

my $cgi = CGI->new;
my $card_number = $cgi->param('cc') || '';

print $cgi->header('text/html');

if ($card_number) {
    validate_card($card_number);
} else {
    show_form();
}

##############################################################################
# Subroutines
##############################################################################

sub show_form {
    print <<HTML;
<!DOCTYPE html>
<html>
<head>
    <title>Credit Card Validator</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 500px; margin: 50px auto; padding: 20px; }
        input[type="text"] { padding: 10px; font-size: 16px; width: 100%; margin: 10px 0; }
        button { padding: 10px 20px; background: #007bff; color: white; border: none; cursor: pointer; }
        .warning { background: #fff3cd; padding: 15px; border-radius: 5px; margin: 20px 0; }
    </style>
</head>
<body>
    <h1>Credit Card Validator</h1>

    <div class="warning">
        <strong>Note:</strong> This only validates number FORMAT using the Luhn algorithm.
        It does NOT process payments or verify card validity with banks.
    </div>

    <form method="POST">
        <label>Enter Credit Card Number:</label>
        <input type="text" name="cc" placeholder="4111111111111111" maxlength="19">
        <button type="submit">Validate</button>
    </form>

    <h3>Test Numbers</h3>
    <p>These are valid format numbers for testing (not real cards):</p>
    <ul>
        <li>Visa: 4111111111111111</li>
        <li>MasterCard: 5500000000000004</li>
        <li>Amex: 340000000000009</li>
        <li>Discover: 6011000000000004</li>
    </ul>
</body>
</html>
HTML
}

sub validate_card {
    my ($number) = @_;

    # Remove spaces and dashes
    $number =~ s/[\s-]//g;

    # Check for non-numeric characters
    unless ($number =~ /^\d+$/) {
        result_page("Invalid", "Card number must contain only digits.", $number);
        return;
    }

    # Check length
    my $len = length($number);
    unless ($len >= 13 && $len <= 19) {
        result_page("Invalid", "Card number must be 13-19 digits.", $number);
        return;
    }

    # Identify card type
    my $card_type = identify_card($number);

    # Luhn algorithm validation
    my $valid = luhn_check($number);

    if ($valid) {
        result_page("Valid", "The number passes Luhn validation.", $number, $card_type);
    } else {
        result_page("Invalid", "The number fails Luhn validation.", $number, $card_type);
    }
}

sub identify_card {
    my ($number) = @_;

    # Visa: Starts with 4
    return "Visa" if $number =~ /^4/;

    # MasterCard: Starts with 51-55 or 2221-2720
    return "MasterCard" if $number =~ /^5[1-5]/ || $number =~ /^2(?:2[2-9][1-9]|[3-6]\d{2}|7[01]\d|720)/;

    # American Express: Starts with 34 or 37
    return "American Express" if $number =~ /^3[47]/;

    # Discover: Starts with 6011, 622126-622925, 644-649, or 65
    return "Discover" if $number =~ /^6(?:011|5|4[4-9]|22(?:1(?:2[6-9]|[3-9]\d)|[2-8]\d{2}|9(?:[01]\d|2[0-5])))/;

    # Diners Club: Starts with 36, 38, or 300-305
    return "Diners Club" if $number =~ /^3(?:0[0-5]|[68])/;

    # JCB: Starts with 35
    return "JCB" if $number =~ /^35/;

    return "Unknown";
}

sub luhn_check {
    my ($number) = @_;

    my @digits = split //, $number;
    my $sum = 0;
    my $alt = 0;

    # Process from right to left
    for my $digit (reverse @digits) {
        if ($alt) {
            $digit *= 2;
            $digit -= 9 if $digit > 9;
        }
        $sum += $digit;
        $alt = !$alt;
    }

    return ($sum % 10) == 0;
}

sub result_page {
    my ($status, $message, $number, $card_type) = @_;
    $card_type ||= 'Unknown';

    my $status_class = $status eq 'Valid' ? 'valid' : 'invalid';
    my $escaped_number = $cgi->escapeHTML($number);

    print <<HTML;
<!DOCTYPE html>
<html>
<head>
    <title>Validation Result</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 500px; margin: 50px auto; padding: 20px; }
        .result { padding: 20px; border-radius: 5px; margin: 20px 0; }
        .valid { background: #d4edda; border: 1px solid #c3e6cb; }
        .invalid { background: #f8d7da; border: 1px solid #f5c6cb; }
        .details { background: #f8f9fa; padding: 15px; margin: 10px 0; }
        a { color: #007bff; }
    </style>
</head>
<body>
    <h1>Validation Result</h1>

    <div class="result $status_class">
        <strong>Status:</strong> $status<br>
        <strong>Message:</strong> $message
    </div>

    <div class="details">
        <strong>Number:</strong> $escaped_number<br>
        <strong>Card Type:</strong> $card_type<br>
        <strong>Length:</strong> @{[length($number)]} digits
    </div>

    <p><a href="?">Test another number</a></p>

    <hr>
    <p><small>
        <strong>Reminder:</strong> This validates FORMAT only using the Luhn algorithm.
        For actual payments, use a payment processor like Stripe, PayPal, or Square.
    </small></p>
</body>
</html>
HTML
}

__END__

=head1 NAME

ccver.pl - Credit Card Number Validator

=head1 SYNOPSIS

  /cgi-bin/ccver.pl?cc=4111111111111111

=head1 DESCRIPTION

Validates credit card numbers using the Luhn algorithm and identifies
the card type based on the number prefix.

=head1 SECURITY WARNING

This script ONLY validates the mathematical format of card numbers.
It does NOT:
- Verify the card exists
- Check available credit/balance
- Detect stolen or fraudulent cards
- Process actual payments

For real payment processing, use established payment gateways.

=head1 LUHN ALGORITHM

The Luhn algorithm (also called mod 10) is a checksum formula used to
validate identification numbers like credit cards. It works by:

1. Starting from the rightmost digit, double every second digit
2. If doubling results in a number > 9, subtract 9
3. Sum all the digits
4. If the sum is divisible by 10, the number is valid

=head1 CARD TYPE PREFIXES

Visa:             4xxx
MasterCard:       51xx-55xx, 2221-2720
American Express: 34xx, 37xx
Discover:         6011, 65xx, 644-649
Diners Club:      36xx, 38xx, 300-305
JCB:              35xx

=head1 MODERN ALTERNATIVES

For payment processing, use established services:

Payment Gateways:
* Stripe (https://stripe.com)
* PayPal (https://paypal.com)
* Square (https://squareup.com)
* Braintree (https://braintreepayments.com)
* Adyen (https://adyen.com)

JavaScript validation libraries:
* card-validator (npm)
* credit-card-type (npm)

Always use HTTPS and PCI-DSS compliant services for handling real card data.

=head1 LICENSE

Artistic License

=cut
