#!/usr/bin/perl
##############################################################################
# Animation                      Version 1.2                                 #
# Copyright 1996                 worldwidemart.com                           #
# Created 09/28/95               Last Modified 11/21/95                      #
# Scripts Archive:               https://www.worldwidemart.com/scripts/          #
##############################################################################
# COPYRIGHT NOTICE                                                           #
# This script is distributed under the Artistic License.                     #
# You may freely use and modify it, keeping this copyright notice intact.    #
##############################################################################
# DESCRIPTION                                                                #
# Creates server-push animation using the multipart/x-mixed-replace MIME     #
# type. Cycles through a series of images to create animation effects.       #
##############################################################################

use strict;
use warnings;

# Configuration
my @images = (
    'image1.gif',
    'image2.gif',
    'image3.gif',
    'image4.gif',
);

my $image_dir  = './images';    # Directory containing images
my $delay      = 500000;        # Delay between frames (microseconds)
my $loops      = 0;             # Number of loops (0 = infinite)
my $image_type = 'image/gif';   # MIME type of images

##############################################################################
# Main Script
##############################################################################

# Select is used for microsecond delays
select(STDOUT);
$| = 1;  # Unbuffered output

# Send multipart header
my $boundary = "----AnimationBoundary" . time();
print "Content-type: multipart/x-mixed-replace;boundary=$boundary\n\n";

# Animation loop
my $loop_count = 0;
while ($loops == 0 || $loop_count < $loops) {
    foreach my $image (@images) {
        my $file = "$image_dir/$image";

        # Read image data
        open(my $fh, '<:raw', $file) or next;
        my $data = do { local $/; <$fh> };
        close($fh);

        # Send image part
        print "--$boundary\n";
        print "Content-type: $image_type\n\n";
        print $data;

        # Delay between frames
        usleep($delay) if defined &usleep;
        select(undef, undef, undef, $delay / 1000000) unless defined &usleep;
    }
    $loop_count++;
}

# Final boundary
print "--$boundary--\n";

__END__

=head1 NAME

animation.pl - Server Push Animation Script

=head1 SYNOPSIS

  <img src="/cgi-bin/animation.pl" alt="Animation">

=head1 DESCRIPTION

Creates server-push animation by sending multiple images using the
multipart/x-mixed-replace MIME type. This was a popular technique
before animated GIFs became standard.

=head1 CONFIGURATION

@images     - Array of image filenames to cycle through
$image_dir  - Directory containing the images
$delay      - Delay between frames in microseconds
$loops      - Number of animation loops (0 = infinite)
$image_type - MIME type of images (usually image/gif)

=head1 BROWSER SUPPORT

Server-push animation was primarily supported by Netscape Navigator.
Modern browsers have limited support. Consider using:
- Animated GIFs
- CSS animations
- JavaScript animations
- Video formats

=head1 MODERN ALTERNATIVES

* CSS @keyframes animations
* JavaScript requestAnimationFrame
* GIF animations
* WebP animated images
* Lottie animations
* Video (MP4, WebM)

=head1 LICENSE

Artistic License

=cut
